/* 
  APATOR AT-WMBUS-16-2  (T1-MODE, 32.768 kbps)
  ESP32-S3 + CC1101
  Updated: 2025-12-07 (DEEP-FILTER + DEDUP + DIAG)
  NOTE: Radio/CC1101/pins/freq 
*/

#include <Arduino.h>
#include <SPI.h>
#include <WiFi.h>
#include <PubSubClient.h>
#include "mbedtls/aes.h"

SPIClass radioSPI(HSPI);

// ------------------------- USER SETTINGS (unchanged) -------------------------
#define FREQ_MODE 1
#define METER_ID 453950

const char* WIFI_SSID = "****";
const char* WIFI_PASS = "*********";

const char* MQTT_SERVER = "*********";
const int   MQTT_PORT   = 1883;
const char* MQTT_USER   = "******";
const char* MQTT_PASSWD = "********";
const char* MQTT_TOPIC  = "wmbus/apator/453950";

WiFiClient espClient;
PubSubClient mqtt(espClient);

// ------------------------- PIN MAP (unchanged) -------------------------
#define PIN_CS   10
#define PIN_SCK  12
#define PIN_MISO 13
#define PIN_MOSI 11
#define PIN_GDO0 5
#define PIN_GDO2 6

// ------------------------- CC1101 SPI HELPERS (unchanged) -------------------------
inline void csLow()  { digitalWrite(PIN_CS, LOW); }
inline void csHigh() { digitalWrite(PIN_CS, HIGH); }
uint8_t ccXfer(uint8_t b) { return radioSPI.transfer(b); }

void ccWrite(uint8_t reg, uint8_t value) {
  csLow();
  radioSPI.beginTransaction(SPISettings(8000000, MSBFIRST, SPI_MODE0));
  ccXfer(reg); ccXfer(value);
  radioSPI.endTransaction();
  csHigh();
}

uint8_t ccRead(uint8_t reg) {
  csLow();
  radioSPI.beginTransaction(SPISettings(8000000, MSBFIRST, SPI_MODE0));
  ccXfer(reg | 0x80);
  uint8_t v = ccXfer(0);
  radioSPI.endTransaction();
  csHigh();
  return v;
}

uint8_t ccStatusRead(uint8_t reg) {
  csLow();
  radioSPI.beginTransaction(SPISettings(8000000, MSBFIRST, SPI_MODE0));
  ccXfer(reg | 0xC0);
  uint8_t v = ccXfer(0);
  radioSPI.endTransaction();
  csHigh();
  return v;
}

void ccStrobe(uint8_t cmd) {
  csLow();
  radioSPI.beginTransaction(SPISettings(8000000, MSBFIRST, SPI_MODE0));
  ccXfer(cmd);
  radioSPI.endTransaction();
  csHigh();
}

// ------------------------- FREQUENCY PROFILES (unchanged) -------------------------
void ccInitApatorT1_86825() {
  ccWrite(0x00, 0x0D);
  ccWrite(0x0D, 0x21); ccWrite(0x0E, 0x61); ccWrite(0x0F, 0xF3);
  ccWrite(0x10, 0x87); ccWrite(0x11, 0x6B); ccWrite(0x12, 0x13);
  ccWrite(0x13, 0x22); ccWrite(0x14, 0xE5); ccWrite(0x15, 0x36);
  ccWrite(0x18, 0x18); ccWrite(0x19, 0x1D); ccWrite(0x1A, 0x1C);
  ccWrite(0x1B, 0xC7); ccWrite(0x1C, 0x00); ccWrite(0x1D, 0x86);
  ccWrite(0x1E, 0x10); ccWrite(0x23, 0xEA);
  delay(5); ccStrobe(0x34);
}

void ccInitApatorT1_86830() {
  ccWrite(0x00, 0x0D);
  ccWrite(0x0D, 0x21); ccWrite(0x0E, 0x62); ccWrite(0x0F, 0x76);
  ccWrite(0x10, 0x87); ccWrite(0x11, 0x6B); ccWrite(0x12, 0x13);
  ccWrite(0x13, 0x22); ccWrite(0x14, 0xE5); ccWrite(0x15, 0x36);
  ccWrite(0x18, 0x18); ccWrite(0x19, 0x1D); ccWrite(0x1A, 0x1C);
  ccWrite(0x1B, 0xC7); ccWrite(0x1C, 0x00); ccWrite(0x1D, 0x86);
  ccWrite(0x1E, 0x10); ccWrite(0x23, 0xEA);
  delay(5); ccStrobe(0x34);
}

void ccInitApatorT1_86835() {
  ccWrite(0x00, 0x0D);
  ccWrite(0x0D, 0x21); ccWrite(0x0E, 0x63); ccWrite(0x0F, 0x09);
  ccWrite(0x10, 0x87); ccWrite(0x11, 0x6B); ccWrite(0x12, 0x13);
  ccWrite(0x13, 0x22); ccWrite(0x14, 0xE5); ccWrite(0x15, 0x3A);
  ccWrite(0x18, 0x18); ccWrite(0x19, 0x1D); ccWrite(0x1A, 0x1C);
  ccWrite(0x1B, 0xC7); ccWrite(0x1C, 0x00); ccWrite(0x1D, 0x86);
  ccWrite(0x1E, 0x10); ccWrite(0x23, 0xEA);
  delay(5); ccStrobe(0x34);
}

// ------------------------- WIFI & MQTT (unchanged) -------------------------
void wifiConnect() {
  WiFi.mode(WIFI_STA);
  WiFi.begin(WIFI_SSID, WIFI_PASS);
  while (WiFi.status() != WL_CONNECTED) delay(200);
}

void mqttEnsure() {
  if (mqtt.connected()) return;
  mqtt.setServer(MQTT_SERVER, MQTT_PORT);
  mqtt.connect("apator_client", MQTT_USER, MQTT_PASSWD);
}

// ------------------------- AES KEY (unchanged) -------------------------
uint8_t apatorKey[16] = {0};  // 16x 0x00

// ------------------------- DEDUP / TIMERS -------------------------
static float last_m3 = -1.0;
static bool last_flow = false;
static bool last_leak = false;
static bool last_tamper = false;
static unsigned long last_publish_ms = 0;
const unsigned long PUBLISH_MIN_INTERVAL_MS = 60UL * 1000UL; // publish at least every 60s even if same

// ------------------------- HELPER: sanity checks -------------------------
bool sanity_checks(float m3, int temp, bool flow, bool leak, bool tamper) {
  if (m3 < 0.0 || m3 > 20000.0) return false;
  if (temp < -30 || temp > 120) return false;
  // other checks could be added
  return true;
}

// ------------------------- TRY DECRYPT AT OFFSET -------------------------
bool tryDecryptAtOffset(uint8_t *raw, int rx, int offset, String &jsonOut) {
  // needs IV(16) + at least 16 bytes cipher (so ctLen >= 16) and multiple of 16
  if (offset < 0) return false;
  if (offset + 16 >= rx) return false;
  int ctLen = rx - offset - 16;
  if (ctLen <= 0) return false;
  if ((ctLen % 16) != 0) return false;

  uint8_t iv[16];
  memcpy(iv, raw + offset, 16);
  uint8_t *cipher = raw + offset + 16;

  uint8_t *plain = (uint8_t*)malloc(ctLen);
  if (!plain) return false;
  memset(plain, 0, ctLen);

  mbedtls_aes_context aes;
  mbedtls_aes_init(&aes);
  if (mbedtls_aes_setkey_dec(&aes, apatorKey, 128) != 0) {
    mbedtls_aes_free(&aes);
    free(plain);
    return false;
  }

  if (mbedtls_aes_crypt_cbc(&aes, MBEDTLS_AES_DECRYPT, ctLen, iv, cipher, plain) != 0) {
    mbedtls_aes_free(&aes);
    free(plain);
    return false;
  }
  mbedtls_aes_free(&aes);

  // need at least bytes we expect 
  if (ctLen < 17) { free(plain); return false; }

  uint32_t vol100 = ((uint32_t)plain[11] << 8) | (uint32_t)plain[12];
  float m3 = vol100 / 100.0;
  int temp = plain[16];
  uint8_t alarm = plain[15];
  bool flow   = alarm & 0x04;
  bool leak   = alarm & 0x02;
  bool tamper = alarm & 0x01;

  bool ok = sanity_checks(m3, temp, flow, leak, tamper);
  if (!ok) { free(plain); return false; }

  jsonOut = "{";
  jsonOut += "\"id\":" + String(METER_ID) + ",";
  jsonOut += "\"m3\":" + String(m3, 2) + ",";
  jsonOut += "\"flow\":" + String(flow ? "true" : "false") + ",";
  jsonOut += "\"temp\":" + String(temp) + ",";
  jsonOut += "\"leak\":" + String(leak ? "true" : "false") + ",";
  jsonOut += "\"tamper\":" + String(tamper ? "true" : "false");
  jsonOut += "}";

  free(plain);
  return true;
}

// ------------------------- HIGH-LEVEL DECODE  -------------------------
bool decodeApatorFlexible(uint8_t *raw, int len, String &jsonOut, int &foundOffset) {
  foundOffset = -1;

  // fast path: try as-is (offset 0)
  if (len >= 32) {
    if (tryDecryptAtOffset(raw, len, 0, jsonOut)) {
      foundOffset = 0;
      return true;
    }
  }

  // otherwise scan for offsets where (len - offset - 16) is multiple of 16 and >=16
  // limit offsets to reasonable range to save CPU (0..len-32)
  for (int off = 1; off <= len - 32; off++) {
    int ctLen = len - off - 16;
    if (ctLen <= 0) continue;
    if ((ctLen % 16) != 0) continue;
    if (tryDecryptAtOffset(raw, len, off, jsonOut)) {
      foundOffset = off;
      return true;
    }
  }

  return false;
}

// ------------------------- SETUP  -------------------------
void setup() {
  Serial.begin(115200);
  delay(300);

  pinMode(PIN_CS, OUTPUT);
  digitalWrite(PIN_CS, HIGH);

  pinMode(PIN_GDO0, INPUT);
  pinMode(PIN_GDO2, INPUT);

  radioSPI.begin(PIN_SCK, PIN_MISO, PIN_MOSI, PIN_CS);

  Serial.println("Connecting WiFi...");
  wifiConnect();
  Serial.print("WiFi OK, IP: "); Serial.println(WiFi.localIP());

  Serial.println("INIT APATOR T1 MODE...");

  if (FREQ_MODE == 1) ccInitApatorT1_86825();
  else if (FREQ_MODE == 2) ccInitApatorT1_86830();
  else ccInitApatorT1_86835();

  mqtt.setClient(espClient);

  Serial.println("READY (APATOR T1 MODE)");
}

// ------------------------- MAIN LOOP -------------------------
void loop() {
  mqtt.loop();
  yield();

  if (!digitalRead(PIN_GDO0)) return;

  uint8_t rx = ccStatusRead(0x3F) & 0x7F;
  if (rx == 0 || rx > 200) {
    ccStrobe(0x36);
    ccStrobe(0x34);
    return;
  }

  uint8_t frame[256];
  for (uint8_t i = 0; i < rx; i++) frame[i] = ccRead(0x3F);

  int rssi = ccRead(0x34);
  int lqi  = ccRead(0x33);

  String hex = "";
  hex.reserve(rx * 2);
  for (uint8_t i = 0; i < rx; i++) {
    if (frame[i] < 16) hex += "0";
    hex += String(frame[i], HEX);
  }

  mqttEnsure();

  // publish RAW (same as before)
  String rawJson = "{";
  rawJson += "\"id\":" + String(METER_ID) + ",";
  rawJson += "\"len\":" + String(rx) + ",";
  rawJson += "\"rssi\":" + String(rssi) + ",";
  rawJson += "\"lqi\":" + String(lqi) + ",";
  rawJson += "\"hex\":\"" + hex + "\"";
  rawJson += "}";
  mqtt.publish(MQTT_TOPIC, rawJson.c_str());
  Serial.print("MQTT RAW: ");
  Serial.println(rawJson);

  // attempt flexible decode (tries offset scan)
  String decodedJson;
  int foundOffset = -1;
  bool ok = decodeApatorFlexible(frame, rx, decodedJson, foundOffset);

  if (ok) {
    // parse minimal fields to decide if we should publish (dedupe)
    // crude parse (we know format) -> extract m3 and flags
    float m3 = 0.0;
    bool flow = false, leak = false, tamper = false;
    int temp = 0;

    // quick parsing - not JSON lib to keep binary small
    int p_m3 = decodedJson.indexOf("\"m3\":");
    if (p_m3 >= 0) {
      int comma = decodedJson.indexOf(',', p_m3);
      String s = decodedJson.substring(p_m3 + 5, comma);
      m3 = s.toFloat();
    }
    int p_flow = decodedJson.indexOf("\"flow\":");
    if (p_flow >= 0) {
      int comma = decodedJson.indexOf(',', p_flow);
      String s = decodedJson.substring(p_flow + 7, comma);
      flow = (s.indexOf("true") >= 0);
    }
    int p_leak = decodedJson.indexOf("\"leak\":");
    if (p_leak >= 0) {
      int comma = decodedJson.indexOf(',', p_leak);
      if (comma < 0) comma = decodedJson.indexOf('}', p_leak);
      String s = decodedJson.substring(p_leak + 7, comma);
      leak = (s.indexOf("true") >= 0);
    }
    int p_tamper = decodedJson.indexOf("\"tamper\":");
    if (p_tamper >= 0) {
      int comma = decodedJson.indexOf('}', p_tamper);
      String s = decodedJson.substring(p_tamper + 9, comma);
      tamper = (s.indexOf("true") >= 0);
    }
    int p_temp = decodedJson.indexOf("\"temp\":");
    if (p_temp >= 0) {
      int comma = decodedJson.indexOf(',', p_temp);
      if (comma < 0) comma = decodedJson.indexOf('}', p_temp);
      String s = decodedJson.substring(p_temp + 7, comma);
      temp = s.toInt();
    }

    unsigned long now = millis();
    bool changed = false;
    if (last_m3 < 0) changed = true;
    else if (fabs(m3 - last_m3) >= 0.01) changed = true;
    else if (flow != last_flow) changed = true;
    else if (leak != last_leak) changed = true;
    else if (tamper != last_tamper) changed = true;
    else if (now - last_publish_ms > PUBLISH_MIN_INTERVAL_MS) changed = true;

    if (changed) {
      mqtt.publish("wmbus/apator/453950/decoded", decodedJson.c_str());
      Serial.print("DECODED OK (off=");
      Serial.print(foundOffset);
      Serial.print("): ");
      Serial.println(decodedJson);

      last_m3 = m3;
      last_flow = flow;
      last_leak = leak;
      last_tamper = tamper;
      last_publish_ms = now;
    } else {
      Serial.println("DECODED (no change) - suppressed publish");
    }

    // If offset != 0, also send small diagnostic note (so we can later inspect)
    if (foundOffset > 0) {
      String diag = "{\"note\":\"decoded_at_offset\",\"offset\":" + String(foundOffset) + ",\"m3\":" + String(m3,2) + "}";
      mqtt.publish("wmbus/apator/453950/diag", diag.c_str());
    }
  } else {
    // didn't decode -> publish small diagnostic (but not too often)
    static unsigned long last_diag_ms = 0;
    unsigned long now = millis();
    if (now - last_diag_ms > 30UL * 1000UL) { // at most once per 30s
      String diag = "{\"note\":\"no_valid_block\",\"len\":" + String(rx) + ",\"hex\":\"" + hex + "\"}";
      mqtt.publish("wmbus/apator/453950/diag", diag.c_str());
      Serial.print("NO VALID BLOCK -> diag published len=");
      Serial.println(rx);
      last_diag_ms = now;
    } else {
      Serial.println("NO VALID BLOCK (diag suppressed)");
    }
  }

  ccStrobe(0x36);
  ccStrobe(0x34);
}
